/*  
 * Copyright (C) 2005 Martin Pischky (mailto:martin@pischky.de)  
 *
 * This file (MemoryTest.java) is part of IntelHexFile.
 *
 * IntelHexFile is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 */

/*
 * $Log: MemoryTest.java,v $
 * Revision 1.1  2005/03/27 12:29:58  pischky
 * added
 *
 */

package org.fremo.ihx.test;

import org.fremo.ihx.ChunkedMemory;
import org.fremo.ihx.InvalidAddressException;
import org.fremo.ihx.Memory;

import junit.framework.TestCase;

/**
 * JUnit test for ChunkedMemory.
 * If you want to test some other Implementation just 
 * subclass and overwrite setUp().
 * @author martin
 */
public class MemoryTest extends TestCase {
    
    Memory mem;

	public static void main(String[] args) {
		junit.swingui.TestRunner.run(MemoryTest.class);
	}

	/*
	 * @see TestCase#setUp()
	 */
	protected void setUp() throws Exception {
		super.setUp();
        mem = new ChunkedMemory();
	}

	/*
	 * @see TestCase#tearDown()
	 */
	protected void tearDown() throws Exception {
        mem = null;
		super.tearDown();
	}

	/**
	 * Constructor for MemoryTest.
	 * @param name
	 */
	public MemoryTest(String name) {
		super(name);
	}

    /* (non-Javadoc)
     * @see junit.framework.TestCase#runTest()
     */
    protected void runTest() throws Throwable {
       super.runTest();
    }
    
    public void testIsDefined() {
        // mem is empty. check this
        assertFalse( mem.isDefined( 0 ) );
        assertFalse( mem.isDefined( 4 ) );
        assertFalse( mem.isDefined( 5434 ) );
        assertFalse( mem.isDefined( 65535 ) );
        try {   mem.isDefined( -1 );
                fail( "mem.isDefined( -1 )" );
        } catch ( InvalidAddressException e ) { /*ok*/ };
        try {   mem.isDefined( 65536 );
                fail( "mem.isDefined( 65536 )" );
        } catch ( InvalidAddressException e ) { /*ok*/ };
        mem.set( 1, 0x55 );
        assertFalse( mem.isDefined( 0 ) );
        assertFalse( mem.isDefined( 2 ) );
        assertTrue( mem.isDefined( 1 ) );
        mem.set( 1050, 0xAA );
        assertFalse( mem.isDefined( 0 ) );
        assertFalse( mem.isDefined( 2 ) );
        assertTrue( mem.isDefined( 1 ) );
        assertFalse( mem.isDefined( 0 ) );
        assertFalse( mem.isDefined( 1051 ) );
        assertTrue( mem.isDefined( 1050 ) );
    }
    
    public void testSetGet() {
        // mem is empty. check this
        assertEquals( Memory.UNDEFINED, mem.get(1) );
        assertEquals( Memory.UNDEFINED, mem.get(1050) );
        mem.set( 1, 0x55 );
        mem.set( 1050, 0xAA );
        assertEquals( 0x55, mem.get(1) );
        assertEquals( 0xAA, mem.get(1050) );
        assertEquals( Memory.UNDEFINED, mem.get(0) );
        assertEquals( Memory.UNDEFINED, mem.get(2) );
        assertEquals( Memory.UNDEFINED, mem.get(23242) );
        mem.set( 67, (short) 255 );
        mem.set( 23, (short) -1 );
        mem.set( 21023, (byte) -5 );
        mem.set( 57353, (byte) 156 );
        assertEquals( 255, mem.get(67) );
        assertEquals( 255, mem.get(23) );
        assertEquals( 251, mem.get(21023) );
        assertEquals( 156, mem.get(57353) );
        try {   mem.get( -1 );
                fail( "mem.get( -1 )" );
        } catch ( InvalidAddressException e ) { /*ok*/ };
        try {   mem.get( 65536 );
                fail( "mem.get( 65536 )" );
        } catch ( InvalidAddressException e ) { /*ok*/ };
    }
    
    public void testUndefine() {
        // mem is empty. check this
        assertEquals( Memory.UNDEFINED, mem.get(1) );
        assertEquals( Memory.UNDEFINED, mem.get(1050) );
        mem.set( 1, 0x55 );
        mem.set( 1050, 0xAA );
        assertEquals( 0x55, mem.get(1) );
        assertEquals( 0xAA, mem.get(1050) );
        mem.undefine( 1050 );
        mem.undefine( 5 );
        assertTrue( mem.isDefined(1) );
        assertFalse( mem.isDefined(1050) );
        assertFalse( mem.isDefined(5) );
    }
    
    public void testNextUsedAddress() {
        // no memory used at all
        assertEquals( -1, mem.nextUsedAddress(0) );
        mem.set( 1050, 0xAA );
        mem.set( 1051, 0xAA );
        mem.set( 1052, 0xAA );
        mem.set( 4050, 0xAA );
        assertEquals( 1050, mem.nextUsedAddress(0) );
        assertEquals( 1050, mem.nextUsedAddress(1050) );
        assertEquals( 1052, mem.nextUsedAddress(1052) );
        assertEquals( 4050, mem.nextUsedAddress(1053) );
        assertEquals( 4050, mem.nextUsedAddress(1099) );
        assertEquals( -1, mem.nextUsedAddress(4051) );
        mem.set( 0, 0x44 );
        assertEquals( 0, mem.nextUsedAddress(0) );
        mem.set( 0xffff, 0x33 );
        assertEquals( 0xffff, mem.nextUsedAddress(7000) );
        mem.set( 0, 0x11 );
        assertEquals( 0, mem.nextUsedAddress(0) );
    }
    
}
